<?php

namespace App\Http\Controllers\Api;

use App\Models\LoyaltySurvey;

use App\Http\Controllers\Controller;
use App\Http\Resources\JsonResponseArrayPaginateResource;
use App\Http\Resources\JsonResponseArrayResource;
use App\Models\LoyaltySurveyQuestion;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Request as FacadesRequest;
use Illuminate\Support\Facades\Validator as FacadesValidator; 


class LoyaltySurveyController extends Controller
{
  public function index()
  {
    $surveys = LoyaltySurvey::with("LoyaltySurveyQuestions")
      ->withCount([
        'LoyaltySurveyQuestions as total_points' => function ($query) {
          $query->select(DB::raw('SUM(points)'));
        },
      ])
      ->orderBy('id', 'desc')
      ->customPaginate();

    return new JsonResponseArrayPaginateResource(
      true,
      'Survey list!',
      $surveys
    );
  }
  public function getSurveyByCategory($id, $customer)
  {
    $surveys = LoyaltySurvey::where('loyalty_survey.published', 1)
      ->where(function ($query) use ($id) {
        $query
          ->where('loyalty_survey.customers', 'like', '%,' . $id . ',%')
          ->orWhere('loyalty_survey.customers', 'like', '[' . $id . ',%')
          ->orWhere('loyalty_survey.customers', 'like', '%,' . $id . ']')
          ->orWhere('loyalty_survey.customers', 'like', '[' . $id . ']')
          ->orWhere('loyalty_survey.customers', 'like', '[1]');
      })
      ->leftJoin('loyaty_survey_results', function ($join) use ($customer) {
        $join
          ->on('loyalty_survey.id', '=', 'loyaty_survey_results.survey_id')
          ->where('loyaty_survey_results.customer', '=', $customer);
      })

      ->select(
        'loyalty_survey.*',
        'loyaty_survey_results.id as result_id',
        'loyaty_survey_results.submitted as submitted',
        'loyaty_survey_results.scored_points as scored_points'
      )
      ->withCount([
        'LoyaltySurveyQuestions as total_points' => function ($query) {
          $query->select(DB::raw('SUM(points)'));
        },
      ])

      ->orderBy('loyalty_survey.created_at', 'desc')
      ->get();

    return new JsonResponseArrayResource(true, 'Survey list!', $surveys);
  }
  public function getSurveyByCategoryNotAttended($id, $customer)
  {
    $hasResults = LoyaltySurvey::whereHas('loyaltySurveyResult', function ($query) use ($customer) {
      $query->where('customer', $customer);
  })->exists();

  if ($hasResults) {
      return new JsonResponseArrayResource(false, 'No surveys found for the given criteria.', []);
  }
  else{
    $surveys = LoyaltySurvey::where('loyalty_survey.published', 1)
    ->where(function ($query) use ($id) {
      $date = Carbon::now('Asia/Kolkata');
      $formattedDate = $date->format('Y-m-d H:i:s'); 
      $query
        ->where('loyalty_survey.customers', 'like', '%,' . $id . ',%')
        ->orWhere('loyalty_survey.customers', 'like', '[' . $id . ',%')
        ->orWhere('loyalty_survey.customers', 'like', '%,' . $id . ']')
        ->orWhere('loyalty_survey.customers', 'like', '[' . $id . ']')
        ->orWhere('loyalty_survey.customers', 'like', '[1]')
        
        ->where('loyalty_survey.start_date','<=',$formattedDate)
        ->where('loyalty_survey.end_date', ">=", $formattedDate);
    })
    ->leftJoin('loyaty_survey_results', function ($join) use ($customer) {
      $join
        ->on('loyalty_survey.id', '=', 'loyaty_survey_results.survey_id')
        ->where('loyaty_survey_results.customer', '=', $customer);
         
        
    })

    ->select(
      'loyalty_survey.*',
      DB::raw('IFNULL(loyaty_survey_results.id, null) as result_id'),
      'loyaty_survey_results.submitted as submitted',
      'loyaty_survey_results.scored_points as scored_points'
    ) 
    ->withCount([
      'LoyaltySurveyQuestions as total_points' => function ($query) {
        $query->select(DB::raw('SUM(points)'));
      },
    ])

    ->orderBy('loyalty_survey.created_at', 'desc')
    ->get();
        
    return new JsonResponseArrayResource(true, 'Survey list!', $surveys);
  }}

  public function store(Request $request)
  {
    $errors = [];
    $errors['error'] = "Something went wrong !";
    $validator = FacadesValidator::make($request->all(), [
      'survey_name' => 'required',
    ]);

    if ($validator->fails()) {
      return response()->json($validator->errors(), 422);
    }
    try {
      DB::beginTransaction();

      $survey_exist = LoyaltySurvey::where('id', $request->id)->first();
      if ($survey_exist) {
        $survey = LoyaltySurvey::find($survey_exist->id);
        $message = "Survey Updated";
      } else {
        $survey = new LoyaltySurvey();
        $message = "Survey Created";
      }
      $survey->survey_name = $request->survey_name;
      $survey->start_date = $request->start_date;
      $survey->end_date = $request->end_date;
      $survey->alert_dates = $request->alert_dates;
      $survey->customers = $request->customers;
      $survey->survey_details = $request->survey_details;
      $survey->published = $request->published ?? 0;
      $survey->active = $request->active ?? 1;
      if ($survey->save()) {
        if (isset($request->questions) && count($request->questions) > 0) {
          foreach ($request->questions as $items) {
            $questionexist = false;
            if (isset($items['id'])) {
              $questionexist = LoyaltySurveyQuestion::where(
                'survey_id',
                $survey->id
              )
                ->where('id', $items['id'])
                ->first();
            }
            if (!$questionexist) {
              $question = new LoyaltySurveyQuestion();
            } else {
              $question = $questionexist;
            }
            $question->survey_id = $survey->id;
            $question->question = $items['question'];
            $question->answer_hints = $items['answer_hints'] ?? null;
            $question->points = $items['points'] ?? null;
            $question->answer_type = $items['answer_type'] ?? null;
            $question->questionDetails = $items['questionDetails'] ?? null;
            $question->answer = $items['answer'] ?? null;
            $question->auto_evaluate = $items['autoEvaluate'] ? 1 : 0;
            $question->max_number_uploads = $items['maxNumberUploads'] ?? 1;
            $question->active = $items['active'] ?? 1;
            if ($questionexist && !$question->update()) {
              DB::rollback();
              return response()->json($errors, 500);
            } elseif (!$questionexist && !$question->save()) {
              DB::rollback();
              return response()->json($errors, 500);
            }
          }
        }
      } else {
        DB::rollback();
        return response()->json($errors, 500);
      }
      DB::commit();
      $survey->questions = LoyaltySurveyQuestion::where(
        'survey_id',
        $survey->id
      )->get();
      return new JsonResponseArrayResource(true, $message, $survey);
    } catch (Exception $ex) {
      DB::rollback();
      return response()->json($ex->getMessage(), 500);
    }
  }

  /**
   * Display the specified resource.
   //  */
  public function show($id)
  {
    $surveys = LoyaltySurvey::with("LoyaltySurveyQuestions")
      ->orderBy('created_at', 'desc')
      ->where('id', $id)
      ->first();
    $surveyTotalPoint = LoyaltySurveyQuestion::where('survey_id', '=', $id)
      ->where('active', '=', '1')
      ->sum('points');
    $surveys['total_point'] = $surveyTotalPoint;

    return new JsonResponseArrayResource(true, 'Survey list!', $surveys);
  }

  // /**
  //  * Show the form for editing the specified resource.
  //  */
  // public function edit(LoyaltySurveyModel $loyaltySurveyModel)
  // {
  //     //
  // }

  /**
   * Update the specified resource in storage.
   */
  // public function update(UpdateLoyaltySurveyModelRequest $request, LoyaltySurveyModel $loyaltySurveyModel)
  // {
  //     //
  // }

  // /**
  //  * Remove the specified resource from storage.
  //  */
  // public function destroy(LoyaltySurveyModel $loyaltySurveyModel)
  // {
  //     //
  // }


  public function getSurveyByCategoryNotSubmitted($id, $customer)
  {
   
    $surveys = LoyaltySurvey::where('loyalty_survey.published', 1)
      ->where(function ($query) use ($id) {
        $date = Carbon::now('Asia/Kolkata');
        $formattedDate = $date->format('Y-m-d H:i:s'); 
        $query
          ->where('loyalty_survey.customers', 'like', '%,' . $id . ',%')
          ->orWhere('loyalty_survey.customers', 'like', '[' . $id . ',%')
          ->orWhere('loyalty_survey.customers', 'like', '%,' . $id . ']')
          ->orWhere('loyalty_survey.customers', 'like', '[' . $id . ']')
          ->orWhere('loyalty_survey.customers', 'like', '[1]')
          
          ->where('loyalty_survey.start_date','<=',$formattedDate)
          ->where('loyalty_survey.end_date', ">=", $formattedDate);
      })
      ->leftJoin('loyaty_survey_results', function ($join) use ($customer) {
        $join
          ->on('loyalty_survey.id', '=', 'loyaty_survey_results.survey_id')
          ->where('loyaty_survey_results.customer', '=', $customer)
          ->where(function ($query) {
            $query->where('loyaty_survey_results.submitted', '!=', 1)
                  ->orWhere('loyaty_survey_results.submitted', '!=', 0);
        });
      })

      ->select(
        'loyalty_survey.*',
        'loyaty_survey_results.id as result_id',
        'loyaty_survey_results.submitted as submitted',
        'loyaty_survey_results.scored_points as scored_points'
      )
      ->withCount([
        'LoyaltySurveyQuestions as total_points' => function ($query) {
          $query->select(DB::raw('SUM(points)'));
        },
      ])

      ->orderBy('loyalty_survey.created_at', 'desc')
      ->get();

    return new JsonResponseArrayResource(true, 'Survey list!', $surveys);
  }

}
