<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\LoyatySurveyResult;
use App\Http\Requests\StoreLoyatySurveyResultRequest;
use App\Http\Requests\UpdateLoyatySurveyResultRequest;
use App\Http\Resources\JsonResponseArrayPaginateResource;
use App\Http\Resources\JsonResponseArrayResource;
use App\Models\LoyaltySurveyResultAnswers;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator as FacadesValidator;
class LoyatySurveyResultController extends Controller
{
  /**
   * Display a listing of the resource.
   */
  public function index()
  {
    $surveys = LoyatySurveyResult::with("LoyaltySurveyAnswers")
      ->withCount([
        'LoyaltySurveyAnswers as scored_points' => function ($query) {
          $query->select(DB::raw('SUM(scored_points)'));
        },
      ])
      ->orderBy('created_at', 'desc')
      ->customPaginate();

    return new JsonResponseArrayPaginateResource(
      true,
      'Survey Result list!',
      $surveys
    );
  }

  public function publishedSurveys($id)
  {
    $surveys = LoyatySurveyResult::where('submitted', 1)
      ->where('survey_id', $id)
      ->leftJoin(
        'customer',
        'loyaty_survey_results.customer',
        '=',
        'customer.code'
      )
      ->select('loyaty_survey_results.*', 'customer.name as name')
      ->orderBy('loyaty_survey_results.submitted_at', 'desc')
      ->customPaginate();

    return new JsonResponseArrayPaginateResource(
      true,
      'Submitted Survey list!',
      $surveys
    );
  }
  public function surveyResultById($id)
  {
    $surveys = LoyatySurveyResult::where('survey_id', $id)
      ->leftJoin(
        'customer',
        'loyaty_survey_results.customer',
        '=',
        'customer.code'
      )
      ->select('loyaty_survey_results.*', 'customer.name as name')
      ->orderBy('loyaty_survey_results.created_at', 'desc')
      ->customPaginate();

    return new JsonResponseArrayPaginateResource(
      true,
      'Survey list!',
      $surveys
    );
  }

  /**
   * Show the form for creating a new resource.
   */
  public function create()
  {
    //
  }

  /**
   * Store a newly created resource in storage.
   */
  public function store(StoreLoyatySurveyResultRequest $request)
  {
    $validator = FacadesValidator::make($request->all(), [
      'customer' => 'required',
    ]);

    if ($validator->fails()) {
      return response()->json($validator->errors(), 422);
    }
    try {
      DB::beginTransaction();

      $survey_exist = LoyatySurveyResult::where('id', $request->id)->first();
      if ($survey_exist) {
        $survey = LoyatySurveyResult::find($survey_exist->id);
        $message = "Survey Result Updated";
      } else {
        $survey = new LoyatySurveyResult();
        $message = "Survey Result Added";
      }
      $survey->customer = $request->customer;
      $survey->survey_id = $request->surveyId;
      $survey->total_points = $request->totalPoints;
      $survey->scored_points = $request->evaluatedBy ? 0 : null;
      $survey->submitted = $request->submitted;
      $survey->submitted_at = $request->submittedAt;
      $survey->review_status = $request->reviewStatus ?? 0;
      $survey->evaluated_by = $request->evaluatedBy ?? null;
      $survey->active = $request->active ?? 1;
      if ($survey->save()) {
        if (isset($request->answerList) && count($request->answerList) > 0) {
          if ($survey_exist) {
            LoyaltySurveyResultAnswers::where(
              'survey_result_id',
              $survey->id
            )->forcedelete();
          }
          foreach ($request->answerList as $items) {
            $answer = new LoyaltySurveyResultAnswers();
            $answer->survey_result_id = $survey->id;
            $answer->question_id = $items['questionId'];
            $answer->answer = $items['answer'] ?? null;
            $answer->scored_points = $items['scoredPoints'] ?? null;
            if ($request->evaluatedBy) {
              $survey->scored_points =
                $survey->scored_points + $items['scoredPoints'] ?? null;
            }
            if (!$answer->save()) {
              DB::rollback();
              return new JsonResponseArrayResource(
                false,
                "Something went wrong!",
                500
              );
            }
          }
        }
      } else {
        DB::rollback();
        return new JsonResponseArrayResource(
          false,
          "Something went wrong!",
          500
        );
      }
      $survey->save();
      DB::commit();
      $survey->answerList = LoyaltySurveyResultAnswers::where(
        'survey_result_id',
        $survey->id
      )->get();
      return new JsonResponseArrayResource(true, $message, $survey);
    } catch (Exception $ex) {
      DB::rollback();
      return response()->json($ex->getMessage(), 500);
    }
  }

  /**
   * Display the specified resource.
   */
  public function show($id)
  {
    $surveys = LoyatySurveyResult::with("LoyaltySurveyAnswers")
      ->where('loyaty_survey_results.id', $id)
      ->leftJoin(
        'loyalty_survey',
        'loyalty_survey.id',
        '=',
        'loyaty_survey_results.survey_id'
      )
      ->select([
        'loyaty_survey_results.id AS id',
        'loyalty_survey.id AS survey_id',
        'loyalty_survey.start_date AS start_date',
        'loyalty_survey.end_date AS end_date',
        'loyaty_survey_results.total_points AS total_points',
        'loyaty_survey_results.submitted AS submitted',
        'loyaty_survey_results.submitted_at AS submitted_at',
        'loyaty_survey_results.evaluated_by AS evaluated_by',
        'loyaty_survey_results.customer AS customer',
        'loyalty_survey.survey_details AS survey_details',
        'loyalty_survey.survey_name AS survey_name',
      ])

      ->first();

    return new JsonResponseArrayResource(true, 'Survey Existing!', $surveys);
  }

  public function loyatySurveyResultById($id)
  {
    $surveys = LoyatySurveyResult::with(
      "LoyaltySurveyAnswersWithReferenceAnswer"
    )
      ->where('loyaty_survey_results.id', $id)
      ->leftJoin(
        'loyalty_survey',
        'loyalty_survey.id',
        '=',
        'loyaty_survey_results.survey_id'
      )
      ->select([
        'loyaty_survey_results.id AS id',
        'loyalty_survey.id AS survey_id',
        'loyalty_survey.start_date AS start_date',
        'loyalty_survey.end_date AS end_date',
        'loyaty_survey_results.total_points AS total_points',
        'loyaty_survey_results.submitted AS submitted',
        'loyaty_survey_results.submitted_at AS submitted_at',
        'loyaty_survey_results.evaluated_by AS evaluated_by',
        'loyaty_survey_results.customer AS customer',
        'loyaty_survey_results.review_status AS review_status',
        'loyalty_survey.survey_details AS survey_details',
        'loyalty_survey.survey_name AS survey_name',
      ])

      ->first();

    return new JsonResponseArrayResource(
      true,
      'Survey With Ref.Answer!',
      $surveys
    );
  }

  /**
   * Show the form for editing the specified resource.
   */
  public function edit(LoyatySurveyResult $loyatySurveyResult)
  {
    //
  }

  /**
   * Update the specified resource in storage.
   */
  public function update(
    UpdateLoyatySurveyResultRequest $request,
    LoyatySurveyResult $loyatySurveyResult
  ) {
    //
  }

  /**
   * Remove the specified resource from storage.
   */
  public function destroy(LoyatySurveyResult $loyatySurveyResult)
  {
    //
  }
}
